# $Id: Manual.pm,v 1.2 2002/12/24 10:30:05 mike Exp $

# Manual.pm - reference manual for the Scott Adams adventure compiler

package Games::ScottAdams::Manual;
use strict;


=head1 NAME

Games::ScottAdams::Manual - The Scott Adams Adventure Compiler Reference Manual


=head1 SYNOPSIS

	# foo.sa - definition file for Scott Adams adventure "foo"

	%room swamp
	dismal swamp
	%exit n meadow
	%exit e edge
	%exit w grove

	%item mud
	Evil smelling mud
	%getdrop mud

	%action take mud
	here mud
	carried bites
	%result
	get mud
	destroy bites
	msg BOY that really hit the spot!


=head1 DESCRIPTION

The Scott Adams compiler, C<sac>, allows you create adventure games in
a straightforward syntax, and compiles them into the format that was
used in the classic Scott Adams adventures - and which is therefore
now understood by C<scottfree> and various other interpreters for
those old games.

If you're running a Linux system, there's a fair chance that you
already have such an interpreter on your system - it's probably called
C<scottfree>, C<ScottCurses>, C<GnomeScott> or something similar.
Certainly Red Hat Linux distributions from 4.0 onwards (and maybe much
earlier) have come with Scott Adams interpreters.

=cut

### should describe what adventure games are

=pod

This manual describes the syntax of the C<sac> file which C<sac>
compiles into Scott Adams format.

All of the examples are taken from Scott Adams' first game, the
classic I<Adventureland> - a game dripping with atmosphere and
nostalgia which I can't recommend highly enough.


=head1 OVERVIEW

Comments may appear anywhere in a C<sac> file, and have no effect on
the compiled adventure.  They are introduced by a hash character
(C<#>) and extend to the end of the line.

Completely blank lines (and those which are completely blank after
comments have been removed) may appear anywhere in the file, and have
no effect on the compiled adventure.

All data is introduced by a I<directive> - that is, a line which
begins with a percent sign (C<%>) immediately followed by a word
specifying which directive is being used.  Common examples include
C<%room>, C<%exit>, C<%item> and C<%action>.  Directive names are
case-insensitive, so that C<%room>, C<%ROOM>, C<%Room> and indeed
C<%rOoM> all mean the same thing.

We describe the directives in four categories, corresponding to the
four fundamental concepts in Scott Adams adventures: the I<rooms>
through which the player moves, the I<items> found in those rooms, the
I<actions> which the player can perform, and I<global parameters>.

With one exception, the order in which directives and their associated
data appear is not significant.  This yields important flexibility in
how the adventure definition file is laid out: for example, all the
rooms may appear together followed by the items, or each room may be
followed by the items which appear in it; items not initially in play
may be listed first or all, or at the end, or after the rooms in which
they will be brought into being during the game.

The one exception to this order-independence is that the order in
which actions appear is significant, because on each turn, each
possible action is considered in the order that appear.  Ordering
issues are discussed in more detail in the section about the
C<%action> directive, but in summary: while the order of actions
relative to other actions is in some cases significant, the position
of actions relative to rooms, items and global parameters is not.
Actions may be moved ahead of and behind rooms, items and global
parameters with impunity.


=head1 ROOMS

The first fundamental concept of Scott Adams adventures is the rooms:
a connnected network of nodes between which the player can move using
the four primary compass directions plus Up and Down.  With typical
topography, after moving north from one room to another, it's possible
to move south back to the first room - but the system does not enforce
this, making it possible to create complex mazes.

Each room in a C<sac> file is identified by a unique name - typically
short, and made up of alphanumerics, possibly with underscores,
although the only restriction enforced is that it may not contain any
whitespace characters (space, tab, I<etc.>)

Apart from its name, a room is defined by a description and a set of
available exits, each exit specifying its destination room.


=head2 %room

	%room chamber
	root chamber under the stump

Creates a new room whose name is the word immediately after the
C<%room> directive, on the same line.  The following lines, up to but
not including the next line that contains a directive, make up the
description of this room, which is what the player sees.  (The name,
by contrast, is used only by C<sac> itself, as an identifying tag when
the room must be referred to when defining an exit, item or action.)

For historical reasons, Scott Adams interpreters such as C<scottfree>
emit the string ``I'm in a '' (or ``You're in a '', if the appropriate
option is specified) before room descriptions, so that the room
defined above would be described as

	I'm in a root chamber under the stump

When this behaviour is not desired, it can be overridden by beginning
the room description with an asterisk (C<*>), which is not printed but
inhibits the automatic initial string.  For example, the room
definition

	%room lake
	*I'm on a narrow ledge by a chasm. Across the chasm is
	the Throne-room

is described to the player simply as

	I'm on a narrow ledge by a chasm. Across the chasm is
	the Throne-room

=head2 %exit

	%exit u stump

Specifies that it's possible to move from the most recently defined
room in the direction indicated by the first argument, and that doing
so takes the player to the destination indicated by the second
argument.  Rooms may have any number of exits from zero to all
six.

The first argument to the C<%room> directive must be one of the single
letters
C<n>,
C<s>,
C<e>,
C<w>,
C<u> or
C<d>
(or their upper-case equivalents), indicating exits in the directions
north, south, east, west, up and down respectively.

The second argument must be the name of a room defined somewhere in
the C<sac> file.  The destination room's definition may be either
previous or subsequent - forward references are just fine.

It's OK for an exit to lead back to the room it came from, and for
more than one exit to lead in the same direction, as in the following
example:

	%room forest
	forest
	%exit n forest
	%exit s forest
	%exit e meadow
	%exit w forest


=head1 ITEMS

The second fundamental concept of Scott Adams adventures is the items:
things that reside in a room, and in some cases can be picked up,
carried around and left in other rooms.  Typically, some of the items
are ``objects'' like axes and keys, while others are ``scenery'' like
trees, signs, I<etc.>

As with rooms, each item in a C<sac> file is identified by a unique
name - typically a short, alphanumeric-plus-underscores name.  Because
the concepts of room and item are so distinct in the Scott Adams
model, it's OK for a room and an item to share the same name.  In fact
this is often the obvious thing to do - for example, consider a
situtation where the player can see a tunnel, then type C<ENTER
TUNNEL> to move inside the tunnel.  In this case, it's natural for
both the tunnel item and the tunnel room to have the name C<tunnel>.

Apart from its name, an item is defined by its location and possibly
by a ``getdrop'' name - see below.

=head2 %item

	%item rubies
	*Pot of RUBIES*

Creates a new item whose name is the word immediately after the
C<%item> directive, on the same line.  The following line is the
description of this item, which is what the player sees.  (The name is
used only as an identifying tag.)

=over 4

=item *

I<Should we allow multi-line object descriptions by reading up to but
not including the next directive line as with room descriptions?>

=back

If the item name begins with an asterisk (C<*>) then it is considered
to be a treasure: it, along with any other treasures, must be
deposited in the treasury (see below) in order to win the game.  The
asterisk is displayed to the user; traditionally, another asterisk
appears at the end of treasure descriptions, but this is not enforced.

=head2 %at

	%at chamber

By default, each item starts the game in the last room defined before
its C<%item> directive; this means that sequences like the following
do The Right Thing:

	%room lake
	*I'm on the shore of a lake
	%item water
	water
	%item fish
	*GOLDEN FISH*

However, in some cases, it may be convenient to define items at some
other point in a C<sac> file - for example, some authors may prefer to
list all rooms together, then all items together.  In such cases,
an item may be relocated to its correct starting room by providing a
C<%at> directive followed by the name of that room:

	%room lake
	*I'm on the shore of a lake
	%room meadow
	sunny meadow
	%item water
	water
	%at lake

Items defined earlier in the C<sac> file than the first C<%room>
directive are by default not in the game when it starts (though they
may subsequently be brought into the game by DROP actions or similar -
see below.)  This can of course be changed with C<%at> directives,
since here as everywhere else, forward references to rooms that have
not yet been defined are OK.

=head2 %nowhere

	%nowhere

Conversely, when defining an item that should not initially be in
play, it may be convenient to place the definition at a point in the
C<sac> file that places it in a room.  In this case, the C<%nowhere>
directive can be used to start it off out of play.  This is
particularly useful if, for example, an item initially in play is
later to be replaced by one that is initially absent:

	%room stump
	damp hollow stump in the swamp
	%item wbottle
	Water in bottle
	%item ebottle
	Empty bottle
	%nowhere
	# will come into play when water is drunk

(Actually, C<%at> and C<%nowhere> are synonyms, so it's possible to
include commands like ``C<%at>'' alone to start an item out of play,
and ``C<%nowhere stump>'' to start it in the room called C<stump>, but
this would be a bit perverse, now, wouldn't it?)

=head2 %getdrop

	%getdrop lamp

Some of the items in a game - those described above as ``objects''
rather than ``scenery'' - can be picked up and dropped.  Rather than
laboriously coding these actions by hand, it's possible to have the
game automatically handle the GET and DROP actions.  In order to do
this, it needs to know the word the user will use to specify the item,
and this is what the C<%getdrop> directive provides:

	%item lamp
	Old fashioned brass lamp
	%getdrop lamp

If no C<%getdrop> name is provided, then it will not be possible for
the player to pick up or drop the item unless explicit actions are
coded to make this possible.

=head1 ACTIONS

The third fundamental concept of Scott Adams adventures is the
actions: things which the player can do, or which can happen to him,
which result in changes to the state of the world.

State consists primarily of the items' locations, but there are also
some boolean flags, integer counters and saved room-numbers.  The
flags are all set to be false at the start of the game; flag number
15 is special, and indicates whether or not it's dark.  If it is, then
the player can't see without a light source.

No-one seems to know for sure how many flags were supported by the
original Scott interpreters, but by inspection, I<Adventureland> uses
flags 1 to 17, missing out flag 6 for some reason, and making only a
single reference to flag 4 (so that it's not really ``used'' in
any meaningful sense.)

=over 4

=item *

I<The only reference to flag 4 is that it's cleared when the axe is
thrown at the bear, misses and breaks the mirror (and it's never
tested anywhere).  Inspection of the other axe-throwing actions
suggests that this is a mistake, and that Scott intended to clear flag
3.  And sure enough, the behaviour if you say C<at bear> twice after
C<throw axe> is wrong: it understands the context-less second C<at
bear> command instead of refusing is and saying ``What?'':>

	Tell me what to do ? throw axe
	In 2 words tell me at what...like: AT TREE

	Tell me what to do ? at bear
	OH NO... Bear dodges... CRASH!

	Tell me what to do ? at bear
	OK, I threw it.
	A voice BOOOOMS out:
	please leave it alone

	Tell me what to do ? at bear
	What?

I<This is not really relevant to C<sac>, but interesting trivia
nevertheless.  It's funny to find someone's bug twenty-two years after
it was created!>

=back

Anyway, C<scottfree> implements 32 flags, and a comment in the source
code says that the author's never seen a game that uses a flag
numbered higher than that.

There are sixteen counters available, and sixteen slots in which room
numbers can be stored.  The latter can be used to implement
sophisticated vehicles and spells which return the player to a room
that was specified earlier - for example, the C<YOHO> spell in
I<Sorceror of Claymorgue Castle>, which moves you first to a
destination, then back to where you first cast it (I think.)

=over 4

=item *

I<Truth is, I'm not at all sure how the
room-number slots are used; this facility is not used at all in
I<Adventureland>, which is the game I'm most familiar with; and
looking at the reverse-engineered I<Claymorgue> actions doesn't help
much.>

=back

There are four other elements of game state: the player's current
room, indications of which of the sixteen counters and room-number
slots are current (since some operations act on the ``current
counter'' and the ``current location slot'') and the number of turns
for which the light source will continue to function.  You don't need
to worry about this stuff much: it's mostly taken care of behind the
scenes.

=head2 %action

	%action open door
	here closed_door
	carried key

Introduces a new action which occurs when the player types a command
equivalent to the one specified.  Equivalent here means using the
specified verb or a synonym together with the specified noun or a
synonym - so depending on how the game is set up, C<UNLOCK PORTAL>
might be equivalent to C<OPEN DOOR>.

The C<%action> directive may optionally be followed on the same line
by a verb alone instead of a verb-noun pair as above; in this case,
the action occurs whenever the user provides any input beginning with
that word - he may provide the verb alone or with any noun.

The lines following the C<%action> directive, up to but not including
the next directive, are conditions, all of which must be satisfied in
order for the results (see below) to happen.  There is no facility for
specifying that conditions should be OR'red together.

Each condition consists of a single-word opcode, followed by zero or
more parameters as required by the opcode.  The following condition
opcodes are supported:

=over 4

=item C<at> I<ROOM>

True if the player's current room is I<ROOM>, which must be the name
of a room defined somewhere in the C<sac> file.

=item C<carried> I<ITEM>

True if the player is carrying I<ITEM>, which must be the name
of an item defined somewhere in the C<sac> file.

=item C<here> I<ITEM>

True if I<ITEM> is in the player's current room.

=item C<accessible> I<ITEM>

True if I<ITEM> is either being carried by the player or in the
player's current room (i.e. if either C<carried ITEM> or C<here
ITEM> is true.)

=item C<exists> I<ITEM>

True if I<ITEM> is in the game (i.e. is not ``nowhere'').

=item C<moved> I<ITEM>

True if I<ITEM> has been moved from its original location.  This
includes the cases where an item initially not in play has been
brought into play or vice versa, and where an item initially carried
has been dropped or vice versa.  This only tests the current
situation, not I<ITEM>'s history - so if I<ITEM> is moved from its
original room, then put back there, this test will return false.

=item C<flag> I<NUM>

True if flag number I<NUM> is set.

=item C<counter_eq> I<NUM>

True if the current counter's value is I<NUM>.  (A different counter
may be nominated as ``current'' by the C<select_counter> action.)

=item C<counter_le> I<NUM>

True if the current counter's value is I<NUM> or less.

=item C<counter_ge> I<NUM>

True if the current counter's value is I<NUM> or more.

=item C<loaded>

True if the player is carrying at least one item.

=back

The sense of the
C<at>,
C<carried>,
C<here>,
C<accessible>,
C<exists>,
C<moved>,
C<flag>,
and
C<loaded>
opcodes may be negated by prefixed them with an exclamation mark
(C<!>).  There is no direct way to test for the negation of the three
counter-related conditions.

=head2 %result

	%result
	destroy closed_door
	drop open_door
	msg It creaks open.

Introduces a sequence of results which occur if the previous
C<%action> or C<%occur> (see below) directive is satisfied.  The lines
following the C<%result> directive, up to, but not including, the next
directive, are the resulting actions, which are executed in sequence.

Each result action consists of a single-word opcode, followed by zero
or more parameters as required by the opcode.  The following condition
opcodes are supported:

=over 4

=item C<get> I<item>

The specified I<item> is put in the player's inventory, unless too
many items are already being carried (Cf. the C<superget> action).
This works even with items that can't be picked up and dropped
otherwise.

=item C<drop> I<item>

The specified I<item> is put in the player's current location,
irrespective of whether it was previous carried, there, elsewhere or
nowhere (out of the game).  This is the standard way to bring into the
game items which begin nowhere.

=item C<moveto> I<room>

Moves to the specified I<room> and displays its description.

=item C<destroy> I<item>

Removes the specified I<item> from the game, irrespective of whether
it was previously carried, in the current location, elsewhere or
already out of the game (in which case it's a no-op).

=item C<set_dark>

Sets flag 15, which indicates darkness.  Exactly equivalent to
C<set_flag 15>.

=item C<clear_dark>

Clears flag 15, which indicates darkness.  Exactly equivalent to
C<clear_flag 15>.

=item C<set_flag> I<number>

Sets flag I<number>.  In general, this is useful only so that
subsequent actions and occurrences can check the value of the flag, so
there are no pre-defined meanings to the flags.  The only flag with a
``built-in'' meaning is number 15 (darkness).

=item C<destroy2> I<item>

Exactly the same as C<destroy>, but implemented using a different
op-code in the compiled game file.

=item C<clear_flag> I<number>

Clears flag I<number>.

=item C<die>

Implements death by printing an ``I am dead'' message, clearing the
darkness flag and moving to the last defined room, which is
conventionally a ``limbo'' room, as in I<Adventureland>'s
``Find right exit and live again!''  This is not a proper, permanent
death: for that, you need the C<game_over> action.

=item C<put> I<item> I<room>

Puts the specified I<item> in the specified I<room>.

=item C<game_over>

Prints ``The game is now over'', waits five seconds and exits.

=item C<look>

Redisplays the description of the current room, the obvious exits and
any visible items.  This is done automatically whenever the player
moves (with the C<moveto> action), C<get>s an item from the current
room, or C<drop>s an item.  So far as I can tell, it need only be done
explicitly when changing the value of the darkness flag.

=item C<score>

Prints the current score, expressed as a mark out of 100, based on how
many treasures have been stored in the treasury location.  This
causes a division-by-zero error if there are no treasures in the game -
i.e. items whose descriptions begin with an asterisk (C<*>).  So games
without treasures, such as Scott Adams's I<Impossible Mission>, should
not provide an action with this result.

=item C<inventory>

Lists the items that the player carrying.

=item C<set_0>

Sets flag 0.  Exactly equivalent to
C<set_flag 0>.

=item C<clear_0>

Clears flag 0.  Exactly equivalent to
C<clear_flag 0>.

=item C<refill_lamp>

I<###>

=item C<clear_screen>

I<###>

=item C<save_game>

I<###>

=item C<swap> I<item> I<item>

I<###>

=item C<continue>

I<###>

=item C<superget> I<item>

I<###>

=item C<put_with> I<item> I<item>

I<###>

=item C<look2>

Exactly the same as C<look>, but implemented using a different
op-code in the compiled game file.

=item C<decrease_counter>

I<###>

=item C<print_counter>

I<###>

=item C<set_counter> I<number>

I<###>

=item C<swap_loc_default>

I<###>

=item C<select_counter> I<number>

I<###>

=item C<add_counter> I<number>

I<###>

=item C<subtract_counter> I<number>

I<###>

=item C<print_noun>

I<###>

=item C<print_noun_nl>

I<###>

=item C<nl>

I<###>

=item C<swap_loc> I<number>

I<###>

=item C<pause>

I<###>

=item C<special> I<number>

I<###>

=back

=head2 %comment

	%comment need key in order to open door

This directive allows a comment to be associated with an action in the
Scott Adams format data file written by C<sac>.  The comment is
attached to the most recently declared action.  Note that this is very
different from the usual kind of comment introduced by the hash
character (C<#>) which is simply discarded by the compiler.

Why would you ever want to use C<%comment>?  Beats me.

=head2 %occur

	%occur 10

Like C<%action>, the C<%occur> directive introduces a sequence of zero
or more conditions which, if fulfilled, will allow some consequences
to result.  The difference is that C<%occur> actions occur
irrespective of what command the player supplies - indeed, they happen
before anything is typed.  They can be used to implement circumstances
such as falling off a ledge if in an appropriately dangerous room
while carrying a particularly heavy item.

If an optional argument is supplied then that argument is the
percentage chance of the occurrence happening when its conditions are
all satisfied; otherwise the chance is 100%.


=head1 GLOBAL PARAMETERS

Finally, we come to the global parameters, a rag-bag of bits and
pieces which affect the game as a whole.  In general, each of the
following directives should appear exactly once: it's an error for any
one of them not to appear at all, and a warning is generated if any is
used more than once.

=head2 %ident

	%ident 1

This simply specifies a number which uniquely identifies the
adventure.  I have read in the C<Definition> file that comes with the
C<scottfree> distribution that this number (and all others in the
Scott Adams file format) is ``apparently 16 bit''.  I don't know how
this is apparent, but it's possible that some interpreters will choke
on numbers larger than 65536 (2^16-1), or maybe even 32767 (2^15-1)
if they interpret the value as signed.  So you should probably pick a
number smaller than this.

Somewhere out there, there should be a register of all Scott Adams
format games, each with a unique identifier number.  Unfortunately, I
don't know if there is one or where it is - please contact me if you
can point me at it (or if you want to start maintaining one!)

Also unfortunately, the uniqueness of the register is already well and
truly broken (although that doesn't mean we should break it more, of
course!)

Adams' original series of twelve adventures uses numbers 1-12
(I<Adventureland> has the coveted number 1, of course!), and the later
I<Sorceror of Claymorgue Castle> is number thirteen.  Unfortunately,
I<Return to Pirate's Island> and I<The Adventures of Buckaroo_Banzai>
are both given number fourteen; and the two Questprobe adventures,
I<The Incredible Hulk> and I<Spiderman> are both number two again (the
same as the original I<Pirate Adventure>.  What a crock.  At least the
I<Adventureland> ``sampler'' that used to be given away for free has
its own number, 65.

To make matters worse, Brian Haworth's series of eleven I<Mysterious
Adventures> re-use the numbers 1-11.  So there are no fewer than four
adventure number two's.  Ho hum.

=head2 %version

	%version 416

Specifies the version of this adventure.  Looks like Adams went
through 416 design iterations before he got I<Adventureland> into a
state he was happy to release.

=head2 %wordlen

	%wordlen 3

Specifies the number of significant letters in each word known to the
game.  Because this is three for I<Adventureland>, all longer words
can be abbreviated to three letters - so the player can type C<CLI
TRE> (or indeed C<CLIMAX TREMENDOUSLY>) instead of C<CLIMB TREE>.

=head2 %maxload

	%maxload 6

Specifies the maximum number of items that the player can carry at
once - if he tries to pick up something else, the interpreter issues a
suitable message.

=head2 %lighttime

	%lighttime 125

Specifies how many turns the light source is good for.  Light is only
used up when the light source is in the game, so, for example if
there's an unlit lamp in the game and a lit lamp initially not in the
game, the light time doesn't start to tick down until the lamp is lit
(i.e. the lit lamp object is brought into the game.)

=head2 %start

	%start forest

Specifies which room the player starts in.

=head2 %treasury

	%treasury stump

Specifies the room in which the player must store treasure for it to
count towards his score.  Remember that treasures are, by definition,
objects whose name begins with an asterisk (C<*>).  The player's score
at any time is defined as the number of treasures that have been
stored in the treasury, divided by the total number of treasures,
multiplied by 100, and rounded to the nearest integer (so that it's
always in the range 0-100.)

The special room-name C<-> may be used to indicate that treasures must
be carried in order to contribute to the score, rather than deposited
in a particular place.

=head2 %lightsource

I<###>

=head2 %nalias

I<###>

=head2 %valias

I<###>

=head2 %include

	%include foo/bar/baz.sac

Includes the contents of the specified file, exactly as though they
were included inline in the SAC file being processed.  Non-absolute
paths are interpreted relative to the file being parsed at that time,
I<not> relative to the working directory.  For, for example, if the
file C<subdir/thrick.sac> is being parsed, and has a line C<%include
frog.sac>, then the file of that name I<in the C<subdir> directory> is
used.


=head1 AUTHOR

Mike Taylor E<lt>mike@miketaylor.org.ukE<gt>

First version Wednesday 11th April 2001.

=cut

1;
